package mod.azure.azurelib.common.api.client.model;

import java.util.Optional;
import java.util.function.BiConsumer;

import mod.azure.azurelib.common.internal.common.AzureLibException;
import mod.azure.azurelib.common.internal.common.cache.AzureLibCache;
import mod.azure.azurelib.common.internal.common.cache.object.BakedGeoModel;
import mod.azure.azurelib.common.internal.common.cache.object.GeoBone;
import mod.azure.azurelib.common.internal.common.constant.DataTickets;
import mod.azure.azurelib.common.internal.common.core.animatable.GeoAnimatable;
import mod.azure.azurelib.common.internal.common.core.animatable.model.CoreGeoModel;
import mod.azure.azurelib.common.internal.common.core.animation.AnimatableManager;
import mod.azure.azurelib.common.internal.common.core.animation.Animation;
import mod.azure.azurelib.common.internal.common.core.animation.AnimationProcessor;
import mod.azure.azurelib.common.internal.common.core.animation.AnimationState;
import mod.azure.azurelib.common.internal.common.core.molang.MolangParser;
import mod.azure.azurelib.common.internal.common.core.molang.MolangQueries;
import mod.azure.azurelib.common.internal.common.core.object.DataTicket;
import mod.azure.azurelib.common.internal.common.loading.object.BakedAnimations;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_1921;
import net.minecraft.class_243;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import mod.azure.azurelib.common.internal.client.renderer.GeoRenderer;
import mod.azure.azurelib.common.internal.client.util.RenderUtils;

/**
 * Base class for all code-based model objects.<br>
 * All models to registered to a {@link GeoRenderer} should be an instance of this or one of its subclasses.
 */
public abstract class GeoModel<T extends GeoAnimatable> implements CoreGeoModel<T> {
	private final AnimationProcessor<T> processor = new AnimationProcessor<>(this);

	private BakedGeoModel currentModel = null;
	private double animTime;
	private double lastGameTickTime;
	private long lastRenderedInstance = -1;

	/**
	 * Returns the resource path for the {@link BakedGeoModel} (model json file) to render based on the provided animatable
	 */
	public abstract class_2960 getModelResource(T animatable);

	/**
	 * Returns the resource path for the texture file to render based on the provided animatable
	 */
	public abstract class_2960 getTextureResource(T animatable);

	/**
	 * Returns the resourcepath for the {@link BakedAnimations} (animation json file) to use for animations based on the provided animatable
	 */
	public abstract class_2960 getAnimationResource(T animatable);

	/**
	 * Override this and return true if AzureLib should crash when attempting to animate the model, but fails to find a bone.<br>
	 * By default, AzureLib will just gracefully ignore a missing bone, which might cause oddities with incorrect models or mismatching variables.<br>
	 */
	public boolean crashIfBoneMissing() {
		return false;
	}

	/**
	 * Gets the default render type for this animatable, to be selected by default by the renderer using it
	 */
	public class_1921 getRenderType(T animatable, class_2960 texture) {
		return class_1921.method_23578(texture);
	}

	@Override
	public final BakedGeoModel getBakedGeoModel(String location) {
		return getBakedModel(new class_2960(location));
	}

	/**
	 * Get the baked geo model object used for rendering from the given resource path
	 */
	public BakedGeoModel getBakedModel(class_2960 location) {
		BakedGeoModel model = AzureLibCache.getBakedModels().get(location);

		if (model == null)
			throw new AzureLibException(location, "Unable to find model");

		if (model != this.currentModel) {
			this.processor.setActiveModel(model);
			this.currentModel = model;
		}

		return this.currentModel;
	}

	/**
	 * Gets a bone from this model by name
	 * 
	 * @param name The name of the bone
	 * @return An {@link Optional} containing the {@link GeoBone} if one matches, otherwise an empty Optional
	 */
	public Optional<GeoBone> getBone(String name) {
		return Optional.ofNullable((GeoBone) getAnimationProcessor().getBone(name));
	}

	/**
	 * Get the baked animation object used for rendering from the given resource path
	 */
	@Override
	public Animation getAnimation(T animatable, String name) {
		class_2960 location = getAnimationResource(animatable);
		BakedAnimations bakedAnimations = AzureLibCache.getBakedAnimations().get(location);

		if (bakedAnimations == null)
			throw new AzureLibException(location, "Unable to find animation.");

		return bakedAnimations.getAnimation(name);
	}

	@Override
	public AnimationProcessor<T> getAnimationProcessor() {
		return this.processor;
	}

	/**
	 * Add additional {@link DataTicket DataTickets} to the {@link AnimationState} to be handled by your animation handler at render time
	 * 
	 * @param animatable   The animatable instance currently being animated
	 * @param instanceId   The unique instance id of the animatable being animated
	 * @param dataConsumer The DataTicket + data consumer to be added to the AnimationEvent
	 */
	public void addAdditionalStateData(T animatable, long instanceId, BiConsumer<DataTicket<T>, T> dataConsumer) {
	}

	@Override
	public void handleAnimations(T animatable, long instanceId, AnimationState<T> animationState) {
		class_310 mc = class_310.method_1551();
		AnimatableManager<T> animatableManager = animatable.getAnimatableInstanceCache().getManagerForId(instanceId);
		Double currentTick = animationState.getData(DataTickets.TICK);

		if (currentTick == null)
			currentTick = animatable instanceof class_1309 livingEntity ? (double) livingEntity.field_6012 : RenderUtils.getCurrentTick();

		if (animatableManager.getFirstTickTime() == -1)
			animatableManager.startedAt(currentTick + mc.method_1488());

		double currentFrameTime = currentTick - animatableManager.getFirstTickTime();
		boolean isReRender = !animatableManager.isFirstTick() && currentFrameTime == animatableManager.getLastUpdateTime();

		if (isReRender && instanceId == this.lastRenderedInstance)
			return;

		if (!isReRender && (!mc.method_1493() || animatable.shouldPlayAnimsWhileGamePaused())) {
			if (animatable instanceof class_1309) {
				animatableManager.updatedAt(currentFrameTime);
			} else {
				animatableManager.updatedAt(currentFrameTime);
			}

			double lastUpdateTime = animatableManager.getLastUpdateTime();
			this.animTime += lastUpdateTime - this.lastGameTickTime;
			this.lastGameTickTime = lastUpdateTime;
		}

		animationState.animationTick = this.animTime;
		AnimationProcessor<T> processor = getAnimationProcessor();

		processor.preAnimationSetup(animationState.getAnimatable(), this.animTime);

		if (!processor.getRegisteredBones().isEmpty())
			processor.tickAnimation(animatable, this, animatableManager, this.animTime, animationState, crashIfBoneMissing());

		setCustomAnimations(animatable, instanceId, animationState);
	}

	@Override
	public void applyMolangQueries(T animatable, double animTime) {
		MolangParser parser = MolangParser.INSTANCE;
		class_310 mc = class_310.method_1551();

		parser.setMemoizedValue(MolangQueries.LIFE_TIME, () -> animTime / 20d);
		parser.setMemoizedValue(MolangQueries.ACTOR_COUNT, mc.field_1687::method_18120);
		parser.setMemoizedValue(MolangQueries.TIME_OF_DAY, () -> mc.field_1687.method_8532() / 24000f);
		parser.setMemoizedValue(MolangQueries.MOON_PHASE, mc.field_1687::method_30273);

		if (animatable instanceof class_1297 entity) {
			parser.setMemoizedValue(MolangQueries.DISTANCE_FROM_CAMERA, () -> mc.field_1773.method_19418().method_19326().method_1022(entity.method_19538()));
			parser.setMemoizedValue(MolangQueries.IS_ON_GROUND, () -> RenderUtils.booleanToFloat(entity.method_24828()));
			parser.setMemoizedValue(MolangQueries.IS_IN_WATER, () -> RenderUtils.booleanToFloat(entity.method_5799()));
			parser.setMemoizedValue(MolangQueries.IS_IN_WATER_OR_RAIN, () -> RenderUtils.booleanToFloat(entity.method_5637()));

			if (entity instanceof class_1309 livingEntity) {
				parser.setMemoizedValue(MolangQueries.HEALTH, livingEntity::method_6032);
				parser.setMemoizedValue(MolangQueries.MAX_HEALTH, livingEntity::method_6063);
				parser.setMemoizedValue(MolangQueries.IS_ON_FIRE, () -> RenderUtils.booleanToFloat(livingEntity.method_5809()));
				parser.setMemoizedValue(MolangQueries.GROUND_SPEED, () -> {
					class_243 velocity = livingEntity.method_18798();

					return class_3532.method_15355((float) ((velocity.field_1352 * velocity.field_1352) + (velocity.field_1350 * velocity.field_1350)));
				});
				parser.setMemoizedValue(MolangQueries.YAW_SPEED, () -> livingEntity.method_5705((float) animTime - livingEntity.method_5705((float) animTime - 0.1f)));
			}
		}
	}
}
