package mod.azure.azurelib.common.api.client.renderer.layer;

import java.util.List;
import java.util.Map;

import org.Vrglab.AzureLib.Utility.Utils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import com.mojang.authlib.GameProfile;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import mod.azure.azurelib.common.api.common.animatable.GeoItem;
import mod.azure.azurelib.common.internal.client.RenderProvider;
import mod.azure.azurelib.common.internal.common.cache.object.BakedGeoModel;
import mod.azure.azurelib.common.internal.common.cache.object.GeoBone;
import mod.azure.azurelib.common.internal.common.cache.object.GeoCube;
import mod.azure.azurelib.common.internal.common.core.animatable.GeoAnimatable;
import net.minecraft.class_1297;
import net.minecraft.class_1304;
import net.minecraft.class_1309;
import net.minecraft.class_1738;
import net.minecraft.class_1747;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1921;
import net.minecraft.class_2190;
import net.minecraft.class_2484;
import net.minecraft.class_2487;
import net.minecraft.class_2631;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_4057;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_4597;
import net.minecraft.class_5598;
import net.minecraft.class_5602;
import net.minecraft.class_572;
import net.minecraft.class_630;
import net.minecraft.class_630.class_628;
import net.minecraft.class_836;
import net.minecraft.class_918;
import mod.azure.azurelib.common.api.client.renderer.GeoArmorRenderer;
import mod.azure.azurelib.common.internal.client.renderer.GeoRenderer;
import mod.azure.azurelib.common.internal.client.util.RenderUtils;

/**
 * Builtin class for handling dynamic armor rendering on AzureLib entities.<br>
 * Supports both {@link GeoItem AzureLib} and {@link net.minecraft.class_1738 Vanilla} armor models.<br>
 * Unlike a traditional armor renderer, this renderer renders per-bone, giving much more flexible armor rendering.
 */
public class ItemArmorGeoLayer<T extends class_1309 & GeoAnimatable> extends GeoRenderLayer<T> {
	protected static final Map<String, class_2960> ARMOR_PATH_CACHE = new Object2ObjectOpenHashMap<>();
	protected static final class_572<class_1309> INNER_ARMOR_MODEL = new class_572<>(class_310.method_1551().method_31974().method_32072(class_5602.field_27579));
	protected static final class_572<class_1309> OUTER_ARMOR_MODEL = new class_572<>(class_310.method_1551().method_31974().method_32072(class_5602.field_27580));

	@Nullable protected class_1799 mainHandStack;
	@Nullable protected class_1799 offhandStack;
	@Nullable protected class_1799 helmetStack;
	@Nullable protected class_1799 chestplateStack;
	@Nullable protected class_1799 leggingsStack;
	@Nullable protected class_1799 bootsStack;

	public ItemArmorGeoLayer(GeoRenderer<T> geoRenderer) {
		super(geoRenderer);
	}

	/**
	 * Return an EquipmentSlot for a given {@link class_1799} and animatable instance.<br>
	 * This is what determines the base model to use for rendering a particular stack
	 */
	@NotNull
	protected class_1304 getEquipmentSlotForBone(GeoBone bone, class_1799 stack, T animatable) {
		for(class_1304 slot : class_1304.values()) {
			if(slot.method_5925() == class_1304.class_1305.field_6178) {
				if(stack == animatable.method_6118(slot))
					return slot;
			}
		}

		return class_1304.field_6174;
	}

	/**
	 * Return a ModelPart for a given {@link GeoBone}.<br>
	 * This is then transformed into position for the final render
	 */
	@NotNull
	protected class_630 getModelPartForBone(GeoBone bone, class_1304 slot, class_1799 stack, T animatable, class_572<?> baseModel) {
		return baseModel.field_3391;
	}

	/**
	 * Get the {@link class_1799} relevant to the bone being rendered.<br>
	 * Return null if this bone should be ignored
	 */
	@Nullable
	protected class_1799 getArmorItemForBone(GeoBone bone, T animatable) {
		return null;
	}

	/**
	 * This method is called by the {@link GeoRenderer} before rendering, immediately after {@link GeoRenderer#preRender} has been called.<br>
	 * This allows for RenderLayers to perform pre-render manipulations such as hiding or showing bones
	 */
	@Override
	public void preRender(class_4587 poseStack, T animatable, BakedGeoModel bakedModel, class_1921 renderType, class_4597 bufferSource,
						  class_4588 buffer, float partialTick, int packedLight, int packedOverlay) {
		this.mainHandStack = animatable.method_6118(class_1304.field_6173);
		this.offhandStack = animatable.method_6118(class_1304.field_6171);
		this.helmetStack = animatable.method_6118(class_1304.field_6169);
		this.chestplateStack = animatable.method_6118(class_1304.field_6174);
		this.leggingsStack = animatable.method_6118(class_1304.field_6172);
		this.bootsStack = animatable.method_6118(class_1304.field_6166);
	}

	/**
	 * This method is called by the {@link GeoRenderer} for each bone being rendered.<br>
	 * This is a more expensive call, particularly if being used to render something on a different buffer.<br>
	 * It does however have the benefit of having the matrix translations and other transformations already applied from render-time.<br>
	 * It's recommended to avoid using this unless necessary.<br>
	 * <br>
	 * The {@link GeoBone} in question has already been rendered by this stage.<br>
	 * <br>
	 * If you <i>do</i> use it, and you render something that changes the {@link class_4588 buffer}, you need to reset it back to the previous buffer
	 * using {@link class_4597#getBuffer} before ending the method
	 */
	@Override
	public void renderForBone(class_4587 poseStack, T animatable, GeoBone bone, class_1921 renderType, class_4597 bufferSource,
							  class_4588 buffer, float partialTick, int packedLight, int packedOverlay) {
		class_1799 armorStack = getArmorItemForBone(bone, animatable);

		if (armorStack == null)
			return;

		if (armorStack.method_7909() instanceof class_1747 blockItem && blockItem.method_7711() instanceof class_2190 skullBlock) {
			renderSkullAsArmor(poseStack, bone, armorStack, skullBlock, bufferSource, packedLight);
		}
		else {
			class_1304 slot = getEquipmentSlotForBone(bone, armorStack, animatable);
			class_572<?> model = getModelForItem(bone, slot, armorStack, animatable);
			class_630 modelPart = getModelPartForBone(bone, slot, armorStack, animatable, model);

			if (!((List<class_628>)Utils.getPrivateFinalStaticField(modelPart, modelPart.getClass(), "cubes")).isEmpty()) {
				poseStack.method_22903();
				poseStack.method_22905(-1, -1, 1);

				if (model instanceof GeoArmorRenderer<?> geoArmorRenderer) {
					prepModelPartForRender(poseStack, bone, modelPart);
					geoArmorRenderer.prepForRender(animatable, armorStack, slot, model);
					geoArmorRenderer.applyBoneVisibilityByPart(slot, modelPart, model);
					geoArmorRenderer.method_2828(poseStack, null, packedLight, packedOverlay, 1, 1, 1, 1);
				}
				else if (armorStack.method_7909() instanceof class_1738) {
					prepModelPartForRender(poseStack, bone, modelPart);
					renderVanillaArmorPiece(poseStack, animatable, bone, slot, armorStack, modelPart, bufferSource, partialTick, packedLight, packedOverlay);
				}

				poseStack.method_22909();
			}
		}
	}

	/**
	 * Renders an individual armor piece base on the given {@link GeoBone} and {@link class_1799}
	 */
	protected <I extends class_1792 & GeoItem> void renderVanillaArmorPiece(class_4587 poseStack, T animatable, GeoBone bone, class_1304 slot, class_1799 armorStack,
															   class_630 modelPart, class_4597 bufferSource, float partialTick, int packedLight, int packedOverlay) {
			class_2960 texture = getVanillaArmorResource(animatable, armorStack, slot, "");
			class_4588 buffer = getArmorBuffer(bufferSource, null, texture, armorStack.method_7958());

			if (armorStack.method_7909() instanceof class_4057 dyable) {
				int color = dyable.method_7800(armorStack);

				modelPart.method_22699(poseStack, buffer, packedLight, packedOverlay, (color >> 16 & 255) / 255f, (color >> 8 & 255) / 255f, (color & 255) / 255f, 1);

				texture = getVanillaArmorResource(animatable, armorStack, slot, "overlay");
				buffer = getArmorBuffer(bufferSource, null, texture, false);
			}

			modelPart.method_22699(poseStack, buffer, packedLight, packedOverlay, 1, 1, 1, 1);
	}

	/**
	 * Returns the standard VertexConsumer for armor rendering from the given buffer source.
	 * @param bufferSource The BufferSource to draw the buffer from
	 * @param renderType The RenderType to use for rendering, or null to use the default
	 * @param texturePath The texture path for the render. May be null if renderType is not null
	 * @param enchanted Whether the render should have an enchanted glint or not
	 * @return The buffer to draw to
	 */
	protected class_4588 getArmorBuffer(class_4597 bufferSource, @Nullable class_1921 renderType, @Nullable class_2960 texturePath, boolean enchanted) {
		if (renderType == null)
			renderType = class_1921.method_25448(texturePath);

		return class_918.method_27952(bufferSource, renderType, false, enchanted);
	}

	/**
	 * Returns a cached instance of a base HumanoidModel that is used for rendering/modelling the provided {@link class_1799}
	 */
	@NotNull
	protected class_572<?> getModelForItem(GeoBone bone, class_1304 slot, class_1799 stack, T animatable) {
		class_572<class_1309> defaultModel = slot == class_1304.field_6172 ? INNER_ARMOR_MODEL : OUTER_ARMOR_MODEL;
		
		return RenderProvider.of(stack).getHumanoidArmorModel(animatable, stack, slot, defaultModel);
	}

	/**
	 * Gets a cached resource path for the vanilla armor layer texture for this armor piece.<br>
	 * Equivalent to {@link net.minecraft.class_970#method_4174 HumanoidArmorLayer.getArmorLocation}
	 */
	public class_2960 getVanillaArmorResource(class_1297 entity, class_1799 stack, class_1304 slot, String type) {
		String domain = "minecraft";
		String path = ((class_1738) stack.method_7909()).method_7686().method_7694();
		String[] materialNameSplit = path.split(":", 2);

		if (materialNameSplit.length > 1) {
			domain = materialNameSplit[0];
			path = materialNameSplit[1];
		}

		if (!type.isBlank())
			type = "_" + type;

		String texture = String.format("%s:textures/models/armor/%s_layer_%d%s.png", domain, path, (slot == class_1304.field_6172 ? 2 : 1), type);
		class_2960 ResourceLocation = ARMOR_PATH_CACHE.get(texture);

		if (ResourceLocation == null) {
			ResourceLocation = new class_2960(texture);
			ARMOR_PATH_CACHE.put(texture, ResourceLocation);
		}

		return ARMOR_PATH_CACHE.computeIfAbsent(texture, class_2960::new);
	}

	/**
	 * Render a given {@link class_2190} as a worn armor piece in relation to a given {@link GeoBone}
	 */
	protected void renderSkullAsArmor(class_4587 poseStack, GeoBone bone, class_1799 stack, class_2190 skullBlock, class_4597 bufferSource, int packedLight) {
		class_2487 stackTag = stack.method_7969();
		GameProfile gameProfile = stackTag != null ? class_2631.method_52589(stackTag) : null;

		class_2484.class_2485 type = skullBlock.method_9327();
		class_5598 model = class_836.method_32160(class_310.method_1551().method_31974()).get(type);
		class_1921 renderType = class_836.method_3578(type, gameProfile);

		poseStack.method_22903();
		RenderUtils.translateAndRotateMatrixForBone(poseStack, bone);
		poseStack.method_22905(1.1875f, 1.1875f, 1.1875f);
		poseStack.method_46416(-0.5f, 0, -0.5f);
		class_836.method_32161(null, 0, 0, poseStack, bufferSource, packedLight, model, renderType);
		poseStack.method_22909();
	}

	/**
	 * Prepares the given {@link class_630} for render by setting its translation, position, and rotation values based on the provided {@link GeoBone}
	 * @param poseStack The PoseStack being used for rendering
	 * @param bone The GeoBone to base the translations on
	 * @param sourcePart The ModelPart to translate
	 */
	protected void prepModelPartForRender(class_4587 poseStack, GeoBone bone, class_630 sourcePart) {
		final GeoCube firstCube = bone.getCubes().get(0);
		final class_628 armorCube = ((List<class_628>)Utils.getPrivateFinalStaticField(sourcePart, sourcePart.getClass(), "cubes")).get(0);
		final double armorBoneSizeX = firstCube.size().method_10216();
		final double armorBoneSizeY = firstCube.size().method_10214();
		final double armorBoneSizeZ = firstCube.size().method_10215();
		final double actualArmorSizeX = Math.abs(armorCube.field_3648 - armorCube.field_3645);
		final double actualArmorSizeY = Math.abs(armorCube.field_3647 - armorCube.field_3644);
		final double actualArmorSizeZ = Math.abs(armorCube.field_3646 - armorCube.field_3643);
		float scaleX = (float)(armorBoneSizeX / actualArmorSizeX);
		float scaleY = (float)(armorBoneSizeY / actualArmorSizeY);
		float scaleZ = (float)(armorBoneSizeZ / actualArmorSizeZ);

		sourcePart.method_2851(-(bone.getPivotX() - ((bone.getPivotX() * scaleX) - bone.getPivotX()) / scaleX),
				-(bone.getPivotY() - ((bone.getPivotY() * scaleY) - bone.getPivotY()) / scaleY),
				(bone.getPivotZ() - ((bone.getPivotZ() * scaleZ) - bone.getPivotZ()) / scaleZ));

		sourcePart.field_3654 = -bone.getRotX();
		sourcePart.field_3675 = -bone.getRotY();
		sourcePart.field_3674 = bone.getRotZ();

		poseStack.method_22905(scaleX, scaleY, scaleZ);
	}
}
