package mod.azure.azurelib.common.api.common.animatable;

import java.util.EnumMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import mod.azure.azurelib.common.internal.common.animatable.SingletonGeoAnimatable;
import mod.azure.azurelib.common.platform.Services;
import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_3218;
import net.minecraft.class_811;
import org.jetbrains.annotations.Nullable;

import com.google.common.base.Suppliers;

import mod.azure.azurelib.common.internal.common.cache.AnimatableIdCache;
import mod.azure.azurelib.common.internal.common.constant.DataTickets;
import mod.azure.azurelib.common.internal.common.core.animatable.GeoAnimatable;
import mod.azure.azurelib.common.internal.common.core.animatable.instance.AnimatableInstanceCache;
import mod.azure.azurelib.common.internal.common.core.animatable.instance.SingletonAnimatableInstanceCache;
import mod.azure.azurelib.common.internal.common.core.animation.AnimatableManager;
import mod.azure.azurelib.common.internal.common.core.animation.ContextAwareAnimatableManager;
import mod.azure.azurelib.common.internal.client.util.RenderUtils;

/**
 * The {@link GeoAnimatable GeoAnimatable} interface specific to {@link net.minecraft.class_1792 Items}. This also applies to armor, as they are just items too.
 */
public interface GeoItem extends SingletonGeoAnimatable {
	String ID_NBT_KEY = "AzureLibID";

	/**
	 * Safety wrapper to distance the client-side code from common code.<br>
	 * This should be cached in your {@link net.minecraft.class_1792 Item} class
	 */
	static Supplier<Object> makeRenderer(GeoItem item) {
		if (Services.PLATFORM.isServerEnvironment())
			return () -> null;

		return Suppliers.memoize(() -> {
			AtomicReference<Object> renderProvider = new AtomicReference<>();
			item.createRenderer(renderProvider::set);
			return renderProvider.get();
		});
	}

	/**
	 * Register this as a synched {@code GeoAnimatable} instance with AzureLib's networking functions
	 * <p>
	 * This should be called inside the constructor of your object.
	 */
	static void registerSyncedAnimatable(GeoAnimatable animatable) {
		SingletonGeoAnimatable.registerSyncedAnimatable(animatable);
	}

	/**
	 * Gets the unique identifying number from this ItemStack's {@link net.minecraft.class_2520 NBT}, or {@link Long#MAX_VALUE} if one hasn't been assigned
	 */
	static long getId(class_1799 stack) {
		class_2487 tag = stack.method_7969();

		if (tag == null)
			return Long.MAX_VALUE;

		return tag.method_10537(ID_NBT_KEY);
	}

	/**
	 * Gets the unique identifying number from this ItemStack's {@link net.minecraft.class_2520 NBT}.<br>
	 * If no ID has been reserved for this stack yet, it will reserve a new id and assign it
	 */
	static long getOrAssignId(class_1799 stack, class_3218 level) {
		class_2487 tag = stack.method_7948();
		long id = tag.method_10537(ID_NBT_KEY);

		if (tag.method_10573(ID_NBT_KEY, class_2520.field_33263))
			return id;

		id = AnimatableIdCache.getFreeId(level);

		tag.method_10544(ID_NBT_KEY, id);

		return id;
	}

	/**
	 * Returns the current age/tick of the animatable instance.<br>
	 * By default this is just the animatable's age in ticks, but this method allows for non-ticking custom animatables to provide their own values
	 * 
	 * @param itemStack The ItemStack representing this animatable
	 * @return The current tick/age of the animatable, for animation purposes
	 */
	@Override
	default double getTick(Object itemStack) {
		return RenderUtils.getCurrentTick();
	}

	/**
	 * Whether this item animatable is perspective aware, handling animations differently depending on the {@link net.minecraft.class_811 render perspective}
	 */
	default boolean isPerspectiveAware() {
		return false;
	}

	/**
	 * Replaces the default AnimatableInstanceCache for GeoItems if {@link GeoItem#isPerspectiveAware()} is true, for perspective-dependent handling
	 */
	@Nullable
	@Override
	default AnimatableInstanceCache animatableCacheOverride() {
		if (isPerspectiveAware())
			return new ContextBasedAnimatableInstanceCache(this);

		return SingletonGeoAnimatable.super.animatableCacheOverride();
	}

	/**
	 * AnimatableInstanceCache specific to GeoItems, for doing render perspective based animations
	 */
	class ContextBasedAnimatableInstanceCache extends SingletonAnimatableInstanceCache {
		public ContextBasedAnimatableInstanceCache(GeoAnimatable animatable) {
			super(animatable);
		}

		/**
		 * Gets an {@link AnimatableManager} instance from this cache, cached under the id provided, or a new one if one doesn't already exist.<br>
		 * This subclass assumes that all animatable instances will be sharing this cache instance, and so differentiates data by ids.
		 */
		@Override
		public AnimatableManager<?> getManagerForId(long uniqueId) {
			if (!this.managers.containsKey(uniqueId))
				this.managers.put(uniqueId, new ContextAwareAnimatableManager<GeoItem, class_811>(this.animatable) {
					@Override
					protected Map<class_811, AnimatableManager<GeoItem>> buildContextOptions(GeoAnimatable animatable) {
						Map<class_811, AnimatableManager<GeoItem>> map = new EnumMap<>(class_811.class);

						for (class_811 context : class_811.values()) {
							map.put(context, new AnimatableManager<>(animatable));
						}

						return map;
					}

					@Override
					public class_811 getCurrentContext() {
						class_811 context = getData(DataTickets.ITEM_RENDER_PERSPECTIVE);

						return context == null ? class_811.field_4315 : context;
					}
				});

			return this.managers.get(uniqueId);
		}
	}
}
