package mod.azure.azurelib.common.internal.common.cache;

import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import mod.azure.azurelib.common.internal.common.core.animatable.model.CoreGeoModel;
import mod.azure.azurelib.common.internal.common.core.animation.Animation;
import mod.azure.azurelib.common.internal.common.AzureLib;
import mod.azure.azurelib.common.internal.common.AzureLibException;
import mod.azure.azurelib.common.internal.common.cache.object.BakedGeoModel;
import mod.azure.azurelib.common.internal.common.loading.FileLoader;
import mod.azure.azurelib.common.internal.common.loading.json.FormatVersion;
import mod.azure.azurelib.common.internal.common.loading.json.raw.Model;
import mod.azure.azurelib.common.internal.common.loading.object.BakedAnimations;
import mod.azure.azurelib.common.internal.common.loading.object.BakedModelFactory;
import mod.azure.azurelib.common.internal.common.loading.object.GeometryTree;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3300;
import net.minecraft.class_3302.class_4045;
import net.minecraft.class_3304;
import net.minecraft.class_3695;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.BiConsumer;
import java.util.function.Function;

/**
 * Cache class for holding loaded
 * {@link Animation Animations} and
 * {@link CoreGeoModel Models}
 */
public final class AzureLibCache {
    private static final Set<String> EXCLUDED_NAMESPACES = ObjectOpenHashSet.of("moreplayermodels", "customnpcs", "gunsrpg");

    private static Map<class_2960, BakedAnimations> ANIMATIONS = Collections.emptyMap();
    private static Map<class_2960, BakedGeoModel> MODELS = Collections.emptyMap();

    public static Map<class_2960, BakedAnimations> getBakedAnimations() {
        if (!AzureLib.hasInitialized)
            throw new AzureLibException("AzureLib was never initialized! Please read the documentation!");

        return ANIMATIONS;
    }

    public static Map<class_2960, BakedGeoModel> getBakedModels() {
        if (!AzureLib.hasInitialized)
            throw new AzureLibException("AzureLib was never initialized! Please read the documentation!");

        return MODELS;
    }

    public static void registerReloadListener() {
        class_310 mc = class_310.method_1551();

        if (mc == null) {
            return;
        }

        if (!(mc.method_1478() instanceof class_3304 resourceManager))
            throw new AzureLibException("AzureLib was initialized too early!");

        resourceManager.method_14477(AzureLibCache::reload);
    }

    public static CompletableFuture<Void> reload(class_4045 stage, class_3300 resourceManager,
                                                 class_3695 preparationsProfiler, class_3695 reloadProfiler, Executor backgroundExecutor,
                                                 Executor gameExecutor) {
        Map<class_2960, BakedAnimations> animations = new Object2ObjectOpenHashMap<>();
        Map<class_2960, BakedGeoModel> models = new Object2ObjectOpenHashMap<>();

        return CompletableFuture
                .allOf(loadAnimations(backgroundExecutor, resourceManager, animations::put),
                        loadModels(backgroundExecutor, resourceManager, models::put))
                .thenCompose(stage::method_18352).thenAcceptAsync(empty -> {
                    AzureLibCache.ANIMATIONS = animations;
                    AzureLibCache.MODELS = models;
                }, gameExecutor);
    }

    private static CompletableFuture<Void> loadAnimations(Executor backgroundExecutor, class_3300 resourceManager,
                                                          BiConsumer<class_2960, BakedAnimations> elementConsumer) {
        return loadResources(backgroundExecutor, resourceManager, "animations",
                resource -> FileLoader.loadAnimationsFile(resource, resourceManager), elementConsumer);
    }

    private static CompletableFuture<Void> loadModels(Executor backgroundExecutor, class_3300 resourceManager,
                                                      BiConsumer<class_2960, BakedGeoModel> elementConsumer) {
        return loadResources(backgroundExecutor, resourceManager, "geo", resource -> {
            Model model = FileLoader.loadModelFile(resource, resourceManager);

            if (model.formatVersion() != FormatVersion.V_1_12_0)
                throw new AzureLibException(resource, "Unsupported geometry json version. Supported versions: 1.12.0");

            return BakedModelFactory.getForNamespace(resource.method_12836())
                    .constructGeoModel(GeometryTree.fromModel(model));
        }, elementConsumer);
    }

    private static <T> CompletableFuture<Void> loadResources(Executor executor, class_3300 resourceManager,
                                                             String type, Function<class_2960, T> loader, BiConsumer<class_2960, T> map) {
        return CompletableFuture.supplyAsync(
                        () -> resourceManager.method_14488(type, fileName -> fileName.toString().endsWith(".json")), executor)
                .thenApplyAsync(resources -> {
                    Map<class_2960, CompletableFuture<T>> tasks = new Object2ObjectOpenHashMap<>();

                    for (class_2960 resource : resources.keySet()) {
                        tasks.put(resource, CompletableFuture.supplyAsync(() -> loader.apply(resource), executor));
                    }

                    return tasks;
                }, executor).thenAcceptAsync(tasks -> {
                    for (Entry<class_2960, CompletableFuture<T>> entry : tasks.entrySet()) {
                        if (!EXCLUDED_NAMESPACES.contains(entry.getKey().method_12836().toLowerCase(Locale.ROOT)))
                            map.accept(entry.getKey(), entry.getValue().join());
                    }
                }, executor);
    }
}
